<?php
// Konfiguracja
$allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'mp4'];
$mediaItems = [];

// Skanowanie katalogu
$files = scandir('.');
foreach ($files as $file) {
    $fileExt = strtolower(pathinfo($file, PATHINFO_EXTENSION));
    if (in_array($fileExt, $allowedExtensions)) {
        $mediaItems[] = [
            'file' => $file,
            'type' => ($fileExt === 'mp4') ? 'video' : 'image'
        ];
    }
}

// Nagłówek HTML
header('Content-Type: text/html; charset=UTF-8');
?>
<!DOCTYPE html>
<html lang="pl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Galeria Zdjęć i Wideo</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: Arial, sans-serif;
            color: #90EE90;
            background-color: #000;
            overflow-x: hidden;
            -webkit-touch-callout: none;
            -webkit-user-select: none;
            -khtml-user-select: none;
            -moz-user-select: none;
            -ms-user-select: none;
            user-select: none;
        }
        .gallery-container {
            width: 100%;
            min-height: 100vh;
            padding: 20px;
        }
        .gallery {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 15px;
            width: 100%;
        }
        .thumbnail-container {
            display: flex;
            justify-content: center;
            align-items: center;
            overflow: hidden;
            position: relative;
        }
        .thumbnail {
            max-width: 100%;
            max-height: 300px;
            width: auto;
            height: auto;
            cursor: pointer;
            transition: all 0.3s ease;
            border: 2px solid #333;
            border-radius: 5px;
            object-fit: contain;
        }
        .video-thumbnail {
            position: relative;
            max-width: 100%;
            max-height: 300px;
        }
        .play-icon {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            color: #90EE90;
            font-size: 50px;
            opacity: 0.8;
            pointer-events: none;
        }
        .thumbnail:hover {
            transform: scale(1.03);
            border-color: #90EE90;
            box-shadow: 0 0 20px rgba(144, 238, 144, 0.4);
        }
        .lightbox {
            display: none;
            position: fixed;
            z-index: 1000;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.95);
            overflow: hidden;
        }
        .lightbox-content {
            position: relative;
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .lightbox-img-container {
            position: relative;
            width: 90%;
            height: 90%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .lightbox-img, .lightbox-video {
            max-width: 100%;
            max-height: 100%;
            width: auto;
            height: auto;
            opacity: 0;
            transition: opacity 0.5s ease-in-out;
            position: absolute;
            object-fit: contain;
        }
        .lightbox-video {
            width: 100%;
            height: 100%;
        }
        .lightbox-img.active, .lightbox-video.active {
            opacity: 1;
        }
        .close {
            position: absolute;
            top: 20px;
            right: 30px;
            color: #90EE90;
            font-size: 40px;
            font-weight: bold;
            cursor: pointer;
            z-index: 1001;
            transition: color 0.3s;
        }
        .close:hover {
            color: #fff;
        }
        .nav-btn {
            color: #90EE90;
            font-size: 60px;
            cursor: pointer;
            padding: 20px;
            user-select: none;
            z-index: 1001;
            position: absolute;
            top: 50%;
            transform: translateY(-50%);
            background-color: rgba(0,0,0,0.3);
            border-radius: 50%;
            transition: all 0.3s;
        }
        .nav-btn:hover {
            background-color: rgba(144, 238, 144, 0.2);
            color: #fff;
        }
        .prev {
            left: 20px;
        }
        .next {
            right: 20px;
        }
        @media (max-width: 768px) {
            .gallery {
                grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
            }
            .thumbnail, .video-thumbnail {
                max-height: 200px;
            }
            .close {
                font-size: 30px;
                top: 15px;
                right: 20px;
            }
            .nav-btn {
                font-size: 40px;
                padding: 15px;
            }
        }
    </style>
</head>
<body>
    <div class="gallery-container">
        <div class="gallery">
            <?php foreach ($mediaItems as $index => $item): ?>
                <div class="thumbnail-container">
                    <?php if ($item['type'] === 'video'): ?>
                        <video class="video-thumbnail thumbnail" onclick="openLightbox(<?= $index ?>)">
                            <source src="<?= htmlspecialchars($item['file']) ?>" type="video/mp4">
                        </video>
                        <div class="play-icon">▶</div>
                    <?php else: ?>
                        <img src="<?= htmlspecialchars($item['file']) ?>" class="thumbnail" onclick="openLightbox(<?= $index ?>)">
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
        </div>
    </div>

    <div id="lightbox" class="lightbox">
        <span class="close" onclick="closeLightbox()">&times;</span>
        <div class="lightbox-content">
            <div class="nav-btn prev" onclick="changeImage(-1)">&#10094;</div>
            <div class="lightbox-img-container">
                <?php foreach ($mediaItems as $index => $item): ?>
                    <?php if ($item['type'] === 'image'): ?>
                        <img src="<?= htmlspecialchars($item['file']) ?>" class="lightbox-img" id="lightbox-img-<?= $index ?>" data-index="<?= $index ?>">
                    <?php else: ?>
                        <video controls class="lightbox-video" id="lightbox-img-<?= $index ?>" data-index="<?= $index ?>">
                            <source src="<?= htmlspecialchars($item['file']) ?>" type="video/mp4">
                        </video>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>
            <div class="nav-btn next" onclick="changeImage(1)">&#10095;</div>
        </div>
    </div>

    <script>
        // Blokada prawego przycisku myszy i źródła strony
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
        
        document.addEventListener('keydown', function(e) {
            if (e.key === 'F12' || (e.ctrlKey && e.shiftKey && e.key === 'I') || (e.ctrlKey && e.shiftKey && e.key === 'J') || (e.ctrlKey && e.key === 'U')) {
                e.preventDefault();
            }
        });

        let currentImageIndex = 0;
        const totalImages = <?= count($mediaItems) ?>;
        const lightbox = document.getElementById('lightbox');
        
        function openLightbox(index) {
            currentImageIndex = index;
            lightbox.style.display = 'block';
            document.body.style.overflow = 'hidden';
            showImage(currentImageIndex);
            
            // Pauza wszystkich filmów przy otwarciu lightboxa
            document.querySelectorAll('.lightbox-video').forEach(video => {
                video.pause();
            });
        }
        
        function closeLightbox() {
            lightbox.style.display = 'none';
            document.body.style.overflow = 'auto';
            hideAllImages();
            
            // Pauza wszystkich filmów przy zamknięciu lightboxa
            document.querySelectorAll('.lightbox-video').forEach(video => {
                video.pause();
            });
        }
        
        function showImage(index) {
            hideAllImages();
            const mediaElement = document.getElementById(`lightbox-img-${index}`);
            if (mediaElement) {
                mediaElement.classList.add('active');
                
                // Jeśli to wideo, odtwórz je
                if (mediaElement.tagName === 'VIDEO') {
                    mediaElement.currentTime = 0;
                    mediaElement.play();
                }
            }
        }
        
        function hideAllImages() {
            document.querySelectorAll('.lightbox-img, .lightbox-video').forEach(media => {
                media.classList.remove('active');
                
                // Jeśli to wideo, zatrzymaj je
                if (media.tagName === 'VIDEO') {
                    media.pause();
                }
            });
        }
        
        function changeImage(step) {
            // Pauza bieżącego wideo przed zmianą
            const currentMedia = document.getElementById(`lightbox-img-${currentImageIndex}`);
            if (currentMedia && currentMedia.tagName === 'VIDEO') {
                currentMedia.pause();
            }
            
            currentImageIndex += step;
            
            if (currentImageIndex >= totalImages) {
                currentImageIndex = 0;
            } else if (currentImageIndex < 0) {
                currentImageIndex = totalImages - 1;
            }
            
            showImage(currentImageIndex);
        }
        
        // Zamknięcie lightboxa po kliknięciu na tło
        lightbox.addEventListener('click', function(e) {
            if (e.target === this) {
                closeLightbox();
            }
        });
        
        // Nawigacja klawiszowa
        document.addEventListener('keydown', function(e) {
            if (lightbox.style.display === 'block') {
                switch(e.key) {
                    case 'ArrowLeft':
                        changeImage(-1);
                        break;
                    case 'ArrowRight':
                        changeImage(1);
                        break;
                    case 'Escape':
                        closeLightbox();
                        break;
                }
            }
        });

        // Detekcja urządzenia i skalowanie
        function isMobileDevice() {
            return (typeof window.orientation !== "undefined") || (navigator.userAgent.indexOf('IEMobile') !== -1);
        }

        function adjustLayout() {
            const thumbnails = document.querySelectorAll('.thumbnail');
            
            thumbnails.forEach(thumb => {
                if (isMobileDevice()) {
                    thumb.style.maxHeight = '200px';
                } else {
                    thumb.style.maxHeight = '300px';
                }
            });
        }

        window.addEventListener('resize', adjustLayout);
        window.addEventListener('load', adjustLayout);
    </script>
</body>
</html>